/*	Copyright (C) 2018-2024 Martin Guy <martinwguy@gmail.com>
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, either version 3 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * calc.c - Do all the heavy calculation of spectra.
 *
 * This file's only entry point is calc(), which is handed a calc_t
 * describing the transform to perform, does the FFT, creates a calc_t
 * and hands it to calc_result(), which sends them via a GUI event
 * to the main loop, which passes them to its calc_notify().
 * The logarithmic frequency axis is applied and coloring done there, not here,
 * so as not to have to recalculate the FFT for zooms, pans and recoloring.
 *
 * When we're finished with the calc_t, it's up to us to free it.
 */

#include "spettro.h"

#include "convert.h"
#include "schedule.h"

#include <time.h>	/* for nanosleep() */

#if ECORE_MAIN
#include <Ecore.h>
#elif SDL_MAIN
#include <SDL.h>
#include <SDL_events.h>
#endif

#include "a_cache.h"
#include "calc.h"
#include "gui.h"	/* For RESULT_EVENT */
#include "lock.h"
#include "spectrum.h"
#include "ui.h"

/*
 * The compute-FFTs function
 */

/* Helper functions */
static void calc_result(calc_t *result);
static calc_t *get_result(calc_t *calc, spectrum *spec, frames_t speclen);

void
calc(calc_t *calc)
{
    spectrum *spec;
    frames_t speclen	= fft_freq_to_speclen(calc->fft_freq,
				      sr);
    calc_t *result;

    /* If parameters have changed since the work was queued, don't bother.
     * This should never happen because we clear the work queue when we
     * change these parameters */
    if (calc->window != window_function || calc->fft_freq != fft_freq) {
	remove_job(calc);
	free(calc);
	return;
    }

    spec = create_spectrum(speclen, calc->window);
    if (spec == NULL) {
	fprintf(stdout, "Can't create spectrum.\n");
	return;
    }

    result = get_result(calc, spec, speclen);

    if (result != NULL) calc_result(result);
    else remove_job(calc);

    destroy_spectrum(spec);
}

/* The function called by calculation threads to report a result */
static void
calc_result(calc_t *result)
{
    /* Send result back to main loop */
#if ECORE_MAIN
    /* Don't return a result if our thread has a cancel request */
    if (ecore_thread_check(result->thread) == FALSE)
	ecore_thread_feedback(result->thread, result);
#elif SDL_MAIN
    SDL_Event event;
    event.type = SDL_USEREVENT;
    event.user.code = RESULT_EVENT;
    event.user.data1 = result;
    while (SDL_PushEvent(&event) != SDL_PUSHEVENT_SUCCESS) {
	struct timespec req;

	/* Sleep for a 1/100th of a second and retry */
	req.tv_sec = 0;
	req.tv_nsec = 1000000000/100;
	nanosleep(&req, NULL);
    }
#endif
}

/*
 * Calculate the magnitude spectrum for a column.
 *
 * speclen is precalculated by the caller from calc->fft_freq
 */
static calc_t *
get_result(calc_t *calc, spectrum *spec, frames_t speclen)
{
        calc_t *result;	/* The result structure */
	frames_t fftsize;
	frames_t r;

	/* Check that the requested sample is within the current interesting
	 * region: either on-screen or in the lookahead region */
	if (calc->t < secs_to_frames(disp_time - (disp_offset - min_x) * secpp) ||
	    calc->t > secs_to_frames(disp_time + (max_x - disp_offset + LOOKAHEAD) * secpp)) {
	    return NULL;
	}

	result = (calc_t *) Malloc(sizeof(calc_t));
	result->t = calc->t;
	result->fft_freq = calc->fft_freq;
	result->window = calc->window;
#if ECORE_MAIN
	result->thread = calc->thread;
#endif

	fftsize = speclen * 2;

	/* Fetch the appropriate audio for our FFT source. */
	/* The data is centred on the requested time. */
	r = read_cached_audio(calc->af, (char *)spec->time_domain, af_float, 1,
			      calc->t - fftsize/2, fftsize);
	if (r < 0) {
	    /* Failed to read the audio cache probably
	     * because it's moved since we started */
	    free(result);
	    return NULL;
	}
	if (r != fftsize) {
	    /* Can happen if they scroll right a lot since we got the work */
	    fprintf(stdout, "Worker got short read from audio cache at %s of %ld frames instead of %ld\n",
		    frames_to_string(result->t), r, fftsize);
	    free(result);
	    return NULL;
	}

	calc_magnitude_spectrum(spec);

	/* We need to pass back a buffer obtained from malloc() that will
	 * subsequently be freed or kept. Rather than memcpy() it, we hijack
	 * the already-allocated buffer and malloc a new one for next time.
	 */
	result->spec = spec->mag_spec;
	spec->mag_spec = Malloc((speclen + 1) * sizeof(*(spec->mag_spec)));

	return(result);
}
