/*	Copyright (C) 2018-2024 Martin Guy <martinwguy@gmail.com>
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, either version 3 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* convert.h: Declarations for convert.c */

#define A4_FREQUENCY	440

/*
 * Vertical position (frequency domain) conversion functions
 */

/* v_pixel_freq_ratio() is frequency ratio between any pixel row
 * and the one above it, used to scroll/zoom by one pixel.
 * e.g. If the graph is 480 pixels high, it's the 479th root of the ratio
 * between the top and bottom frequencies.
 */
#define v_pixel_freq_ratio()		\
    (pow(max_freq / min_freq, 1.0 / (max_y - min_y)))

/* What frequency does the centre of this magnitude index represent? */
#define magindex_to_frequency(k)		\
    (min_freq * pow(max_freq / min_freq, (double)(k) / (maglen - 1)))

#define frequency_to_specindex(freq, sample_rate, speclen) \
    ((freq) * (speclen) / ((sample_rate) / 2))

/* Convert an audio frequency to its index in the magnitude spectrum.
 * To get the screen pixel row it falls in, add min_y. */
#define freq_to_magindex(freq)			\
    round((log(freq) - log(min_freq)) /		\
	  (log(max_freq) - log(min_freq)) *	\
	  (max_y - min_y))

extern char *freq_to_string(freq_t freq);
extern freq_t note_name_to_freq(const char *note);
extern freq_t note_number_to_freq(const int note);

/*
 * Horizontal position (time domain) conversion functions
 */
extern int time_to_piece_column(secs_t t);
extern int time_to_screen_column(secs_t t);
extern int frames_to_piece_column(frames_t t);
extern int frames_to_screen_column(frames_t t);
extern secs_t screen_column_to_start_time(int col);
extern frames_t screen_column_to_start_frames(int col);
extern secs_t frames_to_secs(frames_t frames);
extern frames_t secs_to_frames(secs_t secs);

/*
 * Choose a good FFT size for the given FFT frequency
 */
extern frames_t fft_freq_to_speclen(freq_t fft_freq, freq_t sample_rate);

extern char *seconds_to_string(secs_t secs);
extern char *frames_to_string(frames_t frames);
extern secs_t string_to_seconds(char *string);
